<?php
/**
 * Plugin Name: Smart Title Manager
 * Plugin URI: https://jeffmilner.com/index.php/2025/07/20/introducing-smart-title-manager-the-ultimate-plugin-for-wordpress-title-control/
 * Description: Disable and manage post titles with advanced options including SEO preservation, bulk actions, and conditional hiding
 * Version: 1.0.2
 * Author: Jeff Milner
 * Author URI: https://jeffmilner.com
 * License: GPL v2
 * Text Domain: smart-title-manager
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('STM_PLUGIN_URL', plugin_dir_url(__FILE__));
define('STM_PLUGIN_PATH', plugin_dir_path(__FILE__));

class SmartTitleManager {
    
    private static $instance = null;
    
    public static function getInstance() {
        if (self::$instance == null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        // Admin hooks
        add_action('add_meta_boxes', array($this, 'add_meta_box'));
        add_action('save_post', array($this, 'save_meta_box_data'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // Frontend hooks
        add_filter('the_title', array($this, 'filter_post_title'), 10, 2);
        add_action('wp_head', array($this, 'add_custom_css'));
        add_filter('body_class', array($this, 'add_body_class'));
        
        // Bulk actions
        add_filter('bulk_actions-edit-post', array($this, 'register_bulk_actions'));
        add_filter('handle_bulk_actions-edit-post', array($this, 'handle_bulk_actions'), 10, 3);
        add_action('admin_notices', array($this, 'bulk_action_admin_notice'));
        
        // AJAX handlers
        add_action('wp_ajax_stm_update_hide_settings', array($this, 'ajax_update_hide_settings'));
        
        // Shortcode
        add_shortcode('show_title', array($this, 'show_title_shortcode'));
    }
    
    // Add meta box to post editor
    public function add_meta_box() {
        $post_types = get_post_types(array('public' => true), 'names');
        
        foreach ($post_types as $post_type) {
            add_meta_box(
                'stm_title_options',
                __('Title Display Options', 'smart-title-manager'),
                array($this, 'render_meta_box'),
                $post_type,
                'side',
                'high'
            );
        }
    }
    
    // Render meta box content
    public function render_meta_box($post) {
        wp_nonce_field('stm_meta_box', 'stm_meta_box_nonce');
        
        $hide_title = get_post_meta($post->ID, '_stm_hide_title', true);
        $hide_locations = get_post_meta($post->ID, '_stm_hide_locations', true);
        $custom_css_class = get_post_meta($post->ID, '_stm_custom_css_class', true);
        $seo_title = get_post_meta($post->ID, '_stm_seo_title', true);
        
        if (!is_array($hide_locations)) {
            $hide_locations = array();
        }
        ?>
        <div class="stm-meta-box">
            <p>
                <label>
                    <input type="checkbox" name="stm_hide_title" value="1" <?php checked($hide_title, '1'); ?> />
                    <?php _e('Hide post title', 'smart-title-manager'); ?>
                </label>
            </p>
            
            <div class="stm-hide-options" style="<?php echo $hide_title ? '' : 'display:none;'; ?>">
                <p><strong><?php _e('Hide title in:', 'smart-title-manager'); ?></strong></p>
                <p>
                    <label>
                        <input type="checkbox" name="stm_hide_locations[]" value="single" <?php checked(in_array('single', $hide_locations)); ?> />
                        <?php _e('Single post view', 'smart-title-manager'); ?>
                    </label>
                </p>
                <p>
                    <label>
                        <input type="checkbox" name="stm_hide_locations[]" value="home" <?php checked(in_array('home', $hide_locations)); ?> />
                        <?php _e('Main Blog Page', 'smart-title-manager'); ?>
                    </label>
                </p>
                <p>
                    <label>
                        <input type="checkbox" name="stm_hide_locations[]" value="archive" <?php checked(in_array('archive', $hide_locations)); ?> />
                        <?php _e('Archive pages (Category, Tag, etc.)', 'smart-title-manager'); ?>
                    </label>
                </p>
                <p>
                    <label>
                        <input type="checkbox" name="stm_hide_locations[]" value="search" <?php checked(in_array('search', $hide_locations)); ?> />
                        <?php _e('Search results', 'smart-title-manager'); ?>
                    </label>
                </p>
                <p>
                    <label>
                        <input type="checkbox" name="stm_hide_locations[]" value="feed" <?php checked(in_array('feed', $hide_locations)); ?> />
                        <?php _e('RSS feeds', 'smart-title-manager'); ?>
                    </label>
                </p>
                <p>
                    <label>
                        <strong><?php _e('SEO Title (for meta tags):', 'smart-title-manager'); ?></strong><br>
                        <input type="text" name="stm_seo_title" value="<?php echo esc_attr($seo_title); ?>" style="width:100%;" />
                        <small><?php _e('Leave empty to use original title', 'smart-title-manager'); ?></small>
                    </label>
                </p>
                <p>
                    <label>
                        <strong><?php _e('Custom CSS Class:', 'smart-title-manager'); ?></strong><br>
                        <input type="text" name="stm_custom_css_class" value="<?php echo esc_attr($custom_css_class); ?>" style="width:100%;" />
                        <small><?php _e('Added to body tag when title is hidden', 'smart-title-manager'); ?></small>
                    </label>
                </p>
            </div>
            
            <p class="stm-shortcode-info">
                <small><?php _e('Use [show_title] shortcode to display title anywhere in content', 'smart-title-manager'); ?></small>
            </p>
        </div>
        <?php
    }
    
    // Save meta box data
    public function save_meta_box_data($post_id) {
        if (!isset($_POST['stm_meta_box_nonce']) || !wp_verify_nonce($_POST['stm_meta_box_nonce'], 'stm_meta_box')) {
            return;
        }
        
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        $hide_title = isset($_POST['stm_hide_title']) ? '1' : '';
        update_post_meta($post_id, '_stm_hide_title', $hide_title);
        
        $hide_locations = isset($_POST['stm_hide_locations']) ? array_map('sanitize_text_field', $_POST['stm_hide_locations']) : array();
        update_post_meta($post_id, '_stm_hide_locations', $hide_locations);
        
        if (isset($_POST['stm_custom_css_class'])) {
            update_post_meta($post_id, '_stm_custom_css_class', sanitize_html_class($_POST['stm_custom_css_class']));
        }
        
        if (isset($_POST['stm_seo_title'])) {
            update_post_meta($post_id, '_stm_seo_title', sanitize_text_field($_POST['stm_seo_title']));
        }
    }
    
    // Filter post title on frontend
    public function filter_post_title($title, $id = null) {
        if (!$id || is_admin() || !is_main_query()) {
            return $title;
        }
        
        $hide_title = get_post_meta($id, '_stm_hide_title', true);
        if (!$hide_title) {
            return $title;
        }
        
        $hide_locations = get_post_meta($id, '_stm_hide_locations', true);
        if (!is_array($hide_locations)) {
            $hide_locations = array('single'); // Default to single view
        }
        
        $should_hide = false;
        
        if (is_single() && in_array('single', $hide_locations)) {
            $should_hide = true;
        } elseif (is_home() && in_array('home', $hide_locations)) {
            $should_hide = true;
        } elseif (is_archive() && in_array('archive', $hide_locations)) {
            $should_hide = true;
        } elseif (is_search() && in_array('search', $hide_locations)) {
            $should_hide = true;
        } elseif (is_feed() && in_array('feed', $hide_locations)) {
            $should_hide = true;
        }
        
        if ($should_hide && in_the_loop()) {
            return '';
        }
        
        return $title;
    }
    
    // Add custom CSS to hide title elements
    public function add_custom_css() {
        if (!is_singular()) {
            return;
        }
        
        global $post;
        if (!$post) {
            return;
        }
        
        $hide_title = get_post_meta($post->ID, '_stm_hide_title', true);
        if (!$hide_title) {
            return;
        }
        
        $hide_locations = get_post_meta($post->ID, '_stm_hide_locations', true);
        if (!is_array($hide_locations) || !in_array('single', $hide_locations)) {
            return;
        }
        
        $seo_title = get_post_meta($post->ID, '_stm_seo_title', true);
        if ($seo_title) {
            echo '<meta property="og:title" content="' . esc_attr($seo_title) . '" />' . "\n";
            echo '<meta name="twitter:title" content="' . esc_attr($seo_title) . '" />' . "\n";
        }
        
        ?>
        <style>
            .entry-title, .post-title, .page-title, h1.entry-title, h1.post-title, h1.page-title, .single h1:first-of-type, article > h1:first-child, .content-area h1:first-child { 
                display: none !important; 
            }
            .stm-no-title .site-content { 
                padding-top: 0; 
            }
        </style>
        <?php
    }
    
    // Add body class when title is hidden
    public function add_body_class($classes) {
        if (is_singular()) {
            global $post;
            if ($post) {
                $hide_title = get_post_meta($post->ID, '_stm_hide_title', true);
                if ($hide_title) {
                    $classes[] = 'stm-no-title';
                    
                    $custom_class = get_post_meta($post->ID, '_stm_custom_css_class', true);
                    if ($custom_class) {
                        $classes[] = $custom_class;
                    }
                }
            }
        }
        return $classes;
    }
    
    // Add admin menu
    public function add_admin_menu() {
        add_submenu_page(
            'options-general.php',
            __('Smart Title Manager', 'smart-title-manager'),
            __('Title Manager', 'smart-title-manager'),
            'manage_options',
            'smart-title-manager',
            array($this, 'render_admin_page')
        );
    }
    
    // Render admin page
    public function render_admin_page() {
        ?>
        <div class="wrap">
            <h1><?php _e('Smart Title Manager', 'smart-title-manager'); ?></h1>
            
            <div class="stm-admin-content">
                <h2><?php _e('Posts with Hidden Titles', 'smart-title-manager'); ?></h2>
                
                <?php
                $args = array(
                    'post_type' => get_post_types(array('public' => true), 'names'),
                    'meta_key' => '_stm_hide_title',
                    'meta_value' => '1',
                    'posts_per_page' => -1,
                    'orderby' => 'date',
                    'order' => 'DESC'
                );
                
                $hidden_posts = get_posts($args);
                
                if ($hidden_posts) {
                    ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php _e('Title', 'smart-title-manager'); ?></th>
                                <th><?php _e('Post Type', 'smart-title-manager'); ?></th>
                                <th style="width: 20%;"><?php _e('Hidden In', 'smart-title-manager'); ?></th>
                                <th style="width: 15%;"><?php _e('Actions', 'smart-title-manager'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($hidden_posts as $post) : 
                                $hide_locations = get_post_meta($post->ID, '_stm_hide_locations', true);
                                if (!is_array($hide_locations)) {
                                    $hide_locations = array();
                                }
                            ?>
                                <tr id="stm-row-<?php echo esc_attr($post->ID); ?>">
                                    <td>
                                        <strong>
                                            <a href="<?php echo get_edit_post_link($post->ID); ?>">
                                                <?php echo esc_html($post->post_title); ?>
                                            </a>
                                        </strong>
                                    </td>
                                    <td><?php echo esc_html($post->post_type); ?></td>
                                    <td class="stm-locations-list"><?php echo esc_html(implode(', ', $hide_locations)); ?></td>
                                    <td>
                                        <button class="button button-small stm-modify-settings">
                                            <?php _e('Modify', 'smart-title-manager'); ?>
                                        </button>
                                        <a href="<?php echo get_permalink($post->ID); ?>" target="_blank" class="button button-small">
                                            <?php _e('View', 'smart-title-manager'); ?>
                                        </a>
                                    </td>
                                </tr>
                                <tr class="stm-edit-row" style="display: none;">
                                    <td colspan="4">
                                        <form class="stm-edit-form" data-post-id="<?php echo esc_attr($post->ID); ?>">
                                            <p><strong><?php _e('Hide title in:', 'smart-title-manager'); ?></strong></p>
                                            <div>
                                                <?php
                                                $all_locations = ['single' => 'Single Post', 'home' => 'Blog Page', 'archive' => 'Archives', 'search' => 'Search', 'feed' => 'Feeds'];
                                                foreach ($all_locations as $loc_key => $loc_label) {
                                                    ?>
                                                    <label>
                                                        <input type="checkbox" name="stm_hide_locations[]" value="<?php echo esc_attr($loc_key); ?>" <?php checked(in_array($loc_key, $hide_locations)); ?>>
                                                        <?php echo esc_html($loc_label); ?>
                                                    </label>
                                                    <?php
                                                }
                                                ?>
                                            </div>
                                            <div class="stm-edit-actions">
                                                <button type="submit" class="button button-primary stm-update-button" data-task="update">
                                                    <?php _e('Save Changes', 'smart-title-manager'); ?>
                                                </button>
                                                <button type="button" class="button button-secondary stm-update-button" data-task="show">
                                                    <?php _e('Show Title Everywhere', 'smart-title-manager'); ?>
                                                </button>
                                                <button type="button" class="button button-link stm-cancel-edit">
                                                    <?php _e('Cancel', 'smart-title-manager'); ?>
                                                </button>
                                            </div>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    <?php
                } else {
                    echo '<p>' . __('No posts with hidden titles found.', 'smart-title-manager') . '</p>';
                }
                ?>
                
                <h2><?php _e('Quick Stats', 'smart-title-manager'); ?></h2>
                <div class="stm-stats">
                    <div class="stat-box">
                        <h3><?php echo count($hidden_posts); ?></h3>
                        <p><?php _e('Hidden Titles', 'smart-title-manager'); ?></p>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    // Enqueue admin scripts
    public function enqueue_admin_scripts($hook) {
        if ($hook == 'settings_page_smart-title-manager' || $hook == 'post.php' || $hook == 'post-new.php') {
            wp_enqueue_style(
                'stm-admin-styles',
                STM_PLUGIN_URL . 'admin-styles.css',
                array(),
                '1.0.1' 
            );
            wp_enqueue_script('stm-admin', STM_PLUGIN_URL . 'admin.js', array('jquery'), '1.0.1', true);
            wp_localize_script('stm-admin', 'stm_ajax', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('stm_ajax_nonce')
            ));
        }
    }
    
    // Register bulk actions
    public function register_bulk_actions($bulk_actions) {
        $bulk_actions['stm_hide_titles'] = __('Hide Titles', 'smart-title-manager');
        $bulk_actions['stm_show_titles'] = __('Show Titles', 'smart-title-manager');
        return $bulk_actions;
    }
    
    // Handle bulk actions
    public function handle_bulk_actions($redirect_to, $action, $post_ids) {
        if ($action == 'stm_hide_titles') {
            foreach ($post_ids as $post_id) {
                update_post_meta($post_id, '_stm_hide_title', '1');
                update_post_meta($post_id, '_stm_hide_locations', array('single'));
            }
            $redirect_to = add_query_arg('stm_hidden', count($post_ids), $redirect_to);
        } elseif ($action == 'stm_show_titles') {
            foreach ($post_ids as $post_id) {
                delete_post_meta($post_id, '_stm_hide_title');
                delete_post_meta($post_id, '_stm_hide_locations');
            }
            $redirect_to = add_query_arg('stm_shown', count($post_ids), $redirect_to);
        }
        return $redirect_to;
    }
    
    // Display bulk action notices
    public function bulk_action_admin_notice() {
        if (!empty($_REQUEST['stm_hidden'])) {
            $count = intval($_REQUEST['stm_hidden']);
            printf(
                '<div class="notice notice-success is-dismissible"><p>' .
                _n('Title hidden for %s post.', 'Titles hidden for %s posts.', $count, 'smart-title-manager') .
                '</p></div>',
                $count
            );
        }
        
        if (!empty($_REQUEST['stm_shown'])) {
            $count = intval($_REQUEST['stm_shown']);
            printf(
                '<div class="notice notice-success is-dismissible"><p>' .
                _n('Title shown for %s post.', 'Titles shown for %s posts.', $count, 'smart-title-manager') .
                '</p></div>',
                $count
            );
        }
    }
    
    // AJAX handler for updating settings
    public function ajax_update_hide_settings() {
        check_ajax_referer('stm_ajax_nonce', 'nonce');
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $task = isset($_POST['task']) ? sanitize_key($_POST['task']) : '';

        if (!$post_id || !current_user_can('edit_post', $post_id)) {
            wp_send_json_error('Unauthorized or invalid post.');
        }

        if ($task === 'update') {
            $locations = isset($_POST['locations']) && is_array($_POST['locations']) ? array_map('sanitize_text_field', $_POST['locations']) : array();

            if (empty($locations)) {
                delete_post_meta($post_id, '_stm_hide_title');
                delete_post_meta($post_id, '_stm_hide_locations');
                wp_send_json_success(array(
                    'message' => 'Title shown everywhere.',
                    'locations_html' => '—'
                ));
            } else {
                update_post_meta($post_id, '_stm_hide_title', '1');
                update_post_meta($post_id, '_stm_hide_locations', $locations);
                wp_send_json_success(array(
                    'message' => 'Settings updated.',
                    'locations_html' => esc_html(implode(', ', $locations))
                ));
            }

        } elseif ($task === 'show') {
            delete_post_meta($post_id, '_stm_hide_title');
            delete_post_meta($post_id, '_stm_hide_locations');
            wp_send_json_success(array(
                'message' => 'Title shown everywhere.',
                'locations_html' => '—'
            ));

        } else {
            wp_send_json_error('Invalid task.');
        }
    }
    
    // Shortcode to display title
    public function show_title_shortcode($atts) {
        $atts = shortcode_atts(array(
            'tag' => 'h1',
            'class' => 'entry-title'
        ), $atts);
        
        global $post;
        if (!$post) {
            return '';
        }
    
        remove_filter('the_title', array($this, 'filter_post_title'), 10, 2);
        $raw_title = get_the_title($post->ID);
        add_filter('the_title', array($this, 'filter_post_title'), 10, 2);
        
        $tag = sanitize_key($atts['tag']);
        $class = sanitize_html_class($atts['class']);
        
        return sprintf(
            '<%1$s class="%2$s">%3$s</%1$s>',
            $tag,
            $class,
            esc_html($raw_title)
        );
    }
}

// Initialize the plugin
add_action('plugins_loaded', function() {
    SmartTitleManager::getInstance();
});